import React, { useState } from "react";
import { Pie } from "react-chartjs-2";

import {
  Chart as ChartJS,
  ArcElement,
  Tooltip,
  Legend,
} from "chart.js";

import {
  Gauge,
  User,
  FileText,
  Users,
  Wallet,
  ClipboardList,
  Truck,
  MapPin,
  DollarSign,
  Settings,
  ChevronDown,
} from "lucide-react";

import PendingCustomers from "./customers/PendingCustomers";
import ActiveCustomers from "./customers/ActiveCustomers";
import CustomerPayments from "./payments/CustomerPayments";
import OrderReport from "./reports/OrderReport";
import ShipmentReport from "./reports/ShipmentReport";
import Couriers from "./couriers/Couriers";
import Cities from "./cities/Cities";
import Pricing from "./pricing/Pricing";
import SettingsPage from "./settings/SettingsPage";

import NewOrders from "./requests/NewOrders";
import Karachi from "./requests/Karachi";
import Outstation from "./requests/Outstation";
import Dispatch from "./requests/Dispatch";
import Assigned from "./requests/Assigned";
import Delivered from "./requests/Delivered";
import Returned from "./requests/Returned";
import Cancelled from "./requests/Cancelled";
import PendingOrders from "./requests/PendingOrders";

import Profile from "./components/Profile";
import SearchBar from "./components/SearchBar";

import "./Dashboard.css";
import BookOrders from "./requests/BookedOders";

ChartJS.register(ArcElement, Tooltip, Legend);

export default function Dashboard({ onLogout }) {

  const [activePage, setActivePage] = useState("dashboard");
  const [openRequests, setOpenRequests] = useState(false);

  const [searchValue, setSearchValue] = useState("");
  const [customer, setCustomer] = useState("");

  const handleSearch = () => {
    console.log("Search:", searchValue);
    console.log("Customer:", customer);
  };

  const [counts, setCounts] = useState({
    newOrders: 0,
    karachi: 0,
    outstation: 0,
    dispatch: 0,
    assigned: 0,
    delivered: 0,
    returned: 0,
    cancelled: 0,
    pendingOrders: 0,
  });

  const data = {
    labels: [
      "Pending",
      "Lahore Received",
      "Outstation Received",
      "Dispatch",
      "Assigned",
      "Returned",
      "Delivered",
    ],
    datasets: [
      {
        data: [
          counts.newOrders,
          counts.karachi,
          counts.outstation,
          counts.dispatch,
          counts.assigned,
          counts.returned,
          counts.delivered,
        ],
        backgroundColor: [
          "#1e90ff",
          "#9acd32",
          "#da70d6",
          "#20b2aa",
          "#ff8c00",
          "#ff0000",
          "#32cd32",
        ],
      },
    ],
  };

  return (
    <div className="erp-container">

     
      <aside className="sidebar">
        <h2 className="logo">Courier Admin</h2>

        <ul>
          <li
            className={activePage === "dashboard" ? "active" : ""}
            onClick={() => {
              setActivePage("dashboard");
              setOpenRequests(false);
            }}
          >
            <Gauge size={18} />
            <span>Dashboard</span>
          </li>

          {/* REQUESTS */}
          <li className="has-submenu" onClick={() => setOpenRequests(!openRequests)}>
            <div className="menu-title">
              <FileText size={18} />
              <span>Requests</span>
              <ChevronDown size={16} className={openRequests ? "rotate" : ""} />
            </div>
          </li>

          {openRequests && (
            <ul className="submenu">
              <li onClick={() => setActivePage("newOrders")}>New Orders</li>
              <li onClick={() => setActivePage("karachi")}>Item Received for Lahore</li>
              <li onClick={() => setActivePage("outstation")}>Item Received for Outstation</li>
              <li onClick={() => setActivePage("dispatch")}>Dispatch Orders</li>
              <li onClick={() => setActivePage("assigned")}>Assigned</li>
              <li onClick={() => setActivePage("delivered")}>Delivered</li>
              <li onClick={() => setActivePage("returned")}>Returned</li>
              <li onClick={() => setActivePage("cancelled")}>Cancelled / Not Received</li>
              <li onClick={() => setActivePage("pendingOrders")}>Pending Orders</li>
            </ul>
          )}

          <li onClick={() => setActivePage("pendingCustomers")}><Users size={18}/> Pending Customers</li>
          <li onClick={() => setActivePage("activeCustomers")}><Users size={18}/> Active Customers</li>
          <li onClick={() => setActivePage("customerPayments")}><Wallet size={18}/> Customer Payments</li>
          <li onClick={() => setActivePage("orderReport")}><ClipboardList size={18}/> Order Report</li>
          <li onClick={() => setActivePage("shipmentReport")}><Truck size={18}/> Shipment Report</li>
          <li onClick={() => setActivePage("couriers")}><Truck size={18}/> Couriers</li>
          <li onClick={() => setActivePage("cities")}><MapPin size={18}/> Cities</li>
          <li onClick={() => setActivePage("pricing")}><DollarSign size={18}/> Pricing</li>
          <li onClick={() => setActivePage("settings")}><Settings size={18}/> Setting</li>

          <li
            onClick={onLogout}
            style={{
              marginTop: "20px",
              color: "#ff4d4f",
              cursor: "pointer",
              fontWeight: "600"
            }}
          >
            🚪 Logout
          </li>
        </ul>
      </aside>

   
      <main className="main">

        <SearchBar
          searchValue={searchValue}
          onSearchChange={setSearchValue}
          customer={customer}
          onCustomerChange={setCustomer}
          onSearchSubmit={handleSearch}
        />

        
        <div style={{
          display: "flex",
          justifyContent: "flex-end",
          padding: "10px 20px"
        }}>
          <div
            onClick={() => setActivePage("profile")}
            style={{
              display: "flex",
              alignItems: "center",
              gap: "8px",
              cursor: "pointer",
              fontWeight: "600"
            }}
          >
            <User size={18} /> My Profile
          </div>
        </div>

        {activePage === "dashboard" && (
          <>
            <div className="content-area">
              <div className="left-box-column">
                <div className="info-box brown"><h3>{counts.newOrders}</h3><p>Booked Orders</p></div>
                <div className="info-box orange"><h3>{counts.karachi}</h3><p>Received for Lahore</p></div>
                <div className="info-box green"><h3>{counts.cancelled}</h3><p>Cancelled Orders</p></div>
                <div className="info-box blue"><h3>{counts.assigned}</h3><p>Assigned Orders</p></div>
                <div className="info-box red"><h3>{counts.returned}</h3><p>Returned</p></div>
              </div>

              <div className="chart-section">
                <h2>Orders</h2>
                <Pie data={data} options={{ maintainAspectRatio: false }} />
              </div>
            </div>

            <BookOrders setCounts={setCounts} />
          </>
        )}

        {activePage === "profile" && <Profile />}
        {activePage === "newOrders" && <NewOrders setCounts={setCounts} />}
        {activePage === "karachi" && <Karachi setCounts={setCounts} />}
        {activePage === "outstation" && <Outstation setCounts={setCounts} />}
        {activePage === "dispatch" && <Dispatch setCounts={setCounts} />}
        {activePage === "assigned" && <Assigned setCounts={setCounts} />}
        {activePage === "delivered" && <Delivered setCounts={setCounts} />}
        {activePage === "returned" && <Returned setCounts={setCounts} />}
        {activePage === "cancelled" && <Cancelled setCounts={setCounts} />}
        {activePage === "pendingOrders" && <PendingOrders setCounts={setCounts} />}
        {activePage === "pendingCustomers" && <PendingCustomers />}
        {activePage === "activeCustomers" && <ActiveCustomers />}
        {activePage === "customerPayments" && <CustomerPayments />}
        {activePage === "orderReport" && <OrderReport />}
        {activePage === "shipmentReport" && <ShipmentReport />}
        {activePage === "couriers" && <Couriers />}
        {activePage === "cities" && <Cities />}
        {activePage === "pricing" && <Pricing />}
        {activePage === "settings" && <SettingsPage />}
      </main>
    </div>
  );
}
